# PyDia Code Generation from UML Diagram
# Original Code Copyright (c) 2005  Hans Breuer <hans@breuer.org>

# Eiffel 'renderer' written by Remi Meier <remi_meier@gmx.ch> in 2009

## INSTALL: copy this file into the 'python' folder of your Dia installation.
##          usually this is /usr/share/dia/python/
##          Restart Dia and you should find an export option for Eiffel Code
## Works only with UML diagrams!

#    This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

import sys, dia

class Klass :
	def __init__ (self, name) :
		self.name = name
		self.attributes = []
		# a list, as java/c++ support multiple methods with the same name
		self.operations = []
		self.comment = ""
		self.parents = []
		self.templates = []
		self.inheritance_type = ""
	def AddAttribute(self, name, type, visibility, value, comment) :
		self.attributes.append ((name, (type, visibility, value, comment)))
	def AddOperation(self, name, type, visibility, params, inheritance_type, comment, class_scope) :
		self.operations.append((name,(type, visibility, params, inheritance_type, comment, class_scope)))
	def SetComment(self, s) :
		self.comment = s
	def AddParrent(self, parrent):
		self.parents.append(parrent)
	def AddTemplate(self, template):
		self.templates.append(template)
	def SetInheritance_type(self, inheritance_type):
		self.inheritance_type = inheritance_type

class ObjRenderer :
	"Implements the Object Renderer Interface and transforms diagram into its internal representation"
	def __init__ (self) :
		# an empty dictionary of classes
		self.klasses = {}
		self.arrows = []
		self.filename = ""
		
	def begin_render (self, data, filename) :
		self.filename = filename
		# not only reset the filename but also the other state, otherwise we would accumulate information through every export
		self.klasses = {}
		self.arrows = []
		for layer in data.layers :
			# for the moment ignore layer info. But we could use this to spread accross different files
			for o in layer.objects :
				if o.type.name == "UML - Class" :
					#print o.properties["name"].value
					k = Klass (o.properties["name"].value)
					k.SetComment(o.properties["comment"].value)
					if o.properties["abstract"].value:
						k.SetInheritance_type("abstract")
					if o.properties["template"].value:
						k.SetInheritance_type("template")
					for op in o.properties["operations"].value :
						# op : a tuple with fixed placing, see: objects/UML/umloperations.c:umloperation_props
						# (name, type, comment, stereotype, visibility, inheritance_type, class_scope, params)
						params = []
						for par in op[8] :
							# par : again fixed placement, see objects/UML/umlparameter.c:umlparameter_props
							params.append((par[0], par[1]))
						k.AddOperation (op[0], op[1], op[4], params, op[5], op[2], op[7])
					#print o.properties["attributes"].value
					for attr in o.properties["attributes"].value :
						# see objects/UML/umlattributes.c:umlattribute_props
						#print "\t", attr[0], attr[1], attr[4]
						k.AddAttribute(attr[0], attr[1], attr[4], attr[2], attr[3])
					self.klasses[o.properties["name"].value] = k
					#Connections
				elif o.type.name == "UML - Association" :
					# should already have got attributes relation by names
					pass
				# other UML objects which may be interesting
				# UML - Note, UML - LargePackage, UML - SmallPackage, UML - Dependency, ...
		
		edges = {}
		for layer in data.layers :
			for o in layer.objects :
				for c in o.connections:
					for n in c.connected:
						if not n.type.name in ("UML - Generalization", "UML - Realizes"):
							continue
						if str(n) in edges:
							continue
						edges[str(n)] = None
						if not (n.handles[0].connected_to and n.handles[1].connected_to):
							continue
						par = n.handles[0].connected_to.object
						chi = n.handles[1].connected_to.object
						if not par.type.name == "UML - Class" and chi.type.name == "UML - Class":
							continue
						par_name = par.properties["name"].value
						chi_name = chi.properties["name"].value
						if n.type.name == "UML - Generalization":
							self.klasses[chi_name].AddParrent(par_name)
						else: self.klasses[chi_name].AddTemplate(par_name)
					
	def end_render(self) :
		# without this we would accumulate info from every pass
		self.attributes = {}
		self.operations = {}



class EiffelRenderer(ObjRenderer) :
	def __init__(self) :
		ObjRenderer.__init__(self)
		
	def end_render(self) :
		visibilities = {0:"{ANY}", 2:"{NONE}", 1:"{NONE}"}

		for name, klass in self.klasses.iteritems() :
			f = open(self.filename + "_%s" % name + ".e" , "w")
			
			if klass.inheritance_type == "template": 
				f.write ("class \n\t%s [%s]" % (name, "; ".join(klass.templates)) )
			elif klass.inheritance_type == "abstract":
				f.write ("deferred class \n\t%s" % name)
			else:
				f.write ("class \n\t%s" % name)
			f.write ("\n\n") 
				 
			
			if klass.parents:
				f.write ("\ninherit \n\t%s" % "\n\t".join(klass.parents))
				f.write ("\n\n")
			
			f.write ("\nfeature\n")
			for attrname, (type, visibility, value, comment) in klass.attributes :
				f.write("\t%s : %s " % (attrname, type))
				if value != "":
					f.write(" is %s" % value)
				f.write("\n\t\t-- %s\n\n"% (comment))
			
			
			f.write ("\nfeature\n")
			for methodname, method in klass.operations :
				# if there are no parameter names, something else should appear
				pars = []
				v = ord("a")
				for name, type in method[2]:
					if not name:
						pars.append((type,chr(v)))
						v += 1
					else:
						pars.append((type, name))
						
				pars = "; ".join([name + " : " + type for type, name in pars])
				if pars:
					pars = "(%s)" % pars

				if method[0]: 
					returntype = " : %s" % method[0]
				else:
					returntype = ""
					
				inheritance_type = (method[3]==0 and "deferred ") or "do"
				
				f.write ("\t%s %s%s is\n" % (methodname, pars, returntype))
				f.write ("\t\t\t-- %s\n" % method[4])
				f.write ("\t\t%s\n" % inheritance_type)
				f.write ("\t\tend\n\n\n")
				
			f.write("\n\nend\n\n\n\n")		# for each class
			f.close()
		ObjRenderer.end_render(self)
		
		


# dia-python keeps a reference to the renderer class and uses it on demand
dia.register_export ("PyDia Code Generation (Eiffel)", "e", EiffelRenderer())
